# This script generates all descriptive Figures in the manuscript (Fig. 1, 2, 3, 4, A1)

# Author: Sebastian Hellmeier
# Date: February 12 2021

#load packages
library(ggplot2)
library(tidyverse)
library(lubridate)
library(ggpubr)
library(survival)
library(survminer)

#load weekly data
load("Data/weekly_events_desc.Rda")

##### Figure 1 #####

# transform from wide to long

anti_events <- weekly_events_desc %>% 
  mutate(side = "anti") %>% 
  select(month, side, events = anti_pegida)

pegida_events <- weekly_events_desc %>% 
  mutate(side = "pro") %>% 
  select(month, side, events = pegida)

weekly_activities <- rbind(anti_events, pegida_events)


#Plot demonstrations over time and add lines for Charlie and Cologne and Fall 2015
ggplot(data = weekly_activities, aes(x = month, fill = side)) +
  geom_bar(aes(y = events), stat = "identity", position = "dodge") +
  scale_fill_manual(name = "", labels = c("Anti-Pegida", "Pegida"), values = c("grey80", "grey10")) +
  theme_bw() + 
  scale_x_date(date_breaks = "3 months" , date_labels = "%b-%y") +
  scale_y_continuous(breaks = seq(0, 100, 20)) +
  theme(panel.background = element_rect(fill = "white", colour = "white"),
        panel.grid.major.x = element_blank(),
        panel.grid.minor.x = element_blank(),
        panel.grid.major.y = element_line(size=.1, color="darkgrey")) +
  ylab("Monthly events") + xlab("Month") +
  annotate(geom = "curve", x = as.Date("2015-06-01"), y = 75, xend = as.Date("2015-01-05"), yend = 55.5, 
           curvature = .01, arrow = arrow(length = unit(1, "mm")), lineend = "round") +
  annotate(geom = "curve", x = as.Date("2016-04-01"), y = 33, xend = as.Date("2016-01-09"), yend = 24.5, 
           curvature = .05, arrow = arrow(length = unit(1, "mm")), lineend = "round") +
  annotate(geom = "curve", x = as.Date("2015-08-01"), y = 45, xend = as.Date("2015-09-01"), yend = 12.5, 
           curvature = -.05, arrow = arrow(length = unit(1, "mm")), lineend = "round") +
  annotate(geom = "text", x = as.Date("2015-06-01"), y = 76, label = "Charlie Hebdo shooting", hjust = "left") +
  annotate(geom = "text", x = as.Date("2016-04-11"), y = 33, label = "Cologne sexual assaults", hjust = "left") +
  annotate(geom = "text", x = as.Date("2015-03-01"), y = 48, label = "Halt on Dublin Procedures for Syrians", hjust = "left")

#Export to PDF
ggsave("output/Figure1.pdf", plot = last_plot(), 
       width = 9,
       height = 6,
       dpi = 600)

##### Figure 2 #####
#Clear environment
rm(list=ls())

# load data
load("data/pegpad_waves.Rda")

# subset to the city of Karlsruhe
karlsruhe <- pegida_waves_final_rep %>% 
  filter(id == 43 & pegida_demos > 0) %>% 
  complete(week_date = seq.Date(as.Date("2014-10-27"), as.Date("2015-07-15"), by="week")) %>% 
  fill(id,city, .direction="up")   %>% 
  fill(id,city, .direction="down")   %>% 
  mutate(participants_pegida_total = ifelse(is.na(participants_pegida_total_mean),0,participants_pegida_total_mean),
         spell_start = ifelse(week_date=="2015-02-16",1,0),
         spell_end = ifelse(week_date=="2015-05-25",1,0))

# Plot protests in Karlsruhe over time
ggplot() + geom_step(data = karlsruhe[karlsruhe$week_date>="2015-02-16" & karlsruhe$week_date<="2015-05-25",], mapping = aes(week_date, participants_pegida_total)) +
  geom_line(data = karlsruhe[karlsruhe$week_date<"2015-02-23",], mapping = aes(week_date, participants_pegida_total), linetype="longdash", color="black")  +
  geom_line(data = karlsruhe[karlsruhe$week_date>"2015-05-18",], mapping = aes(week_date, participants_pegida_total), linetype="dotted", color="black")   +
  geom_point(data = karlsruhe[karlsruhe$week_date=="2015-02-16",], mapping = aes(week_date, spell_start), shape = 1, size=6, color="black") + 
  geom_point(data = karlsruhe[karlsruhe$week_date=="2015-05-25",], mapping = aes(week_date, spell_end), shape = 13, size=6,  color="black") + 
  annotate("text", x = as.Date("2015-02-09"), y = 15, label="onset (1)", size = 3.8)  + 
  annotate("text", x=as.Date("2015-06-23"), y = 15, label="demobilization (3)", size = 3.8) +
  theme_classic() + scale_x_date(date_breaks = "months" ,date_labels = "%b-%y") + scale_y_continuous(breaks = seq(0,300,50)) +
  xlab("") + ylab("Pegida protest participants") + 
  geom_bracket(
    xmin = as.Date("2015-02-16"), xmax = as.Date("2015-05-25"), y.position = 250,
    label = "heightened contention (2)", tip.length = 1/20, size = .5) +
  theme(panel.border = element_rect(colour = "black", fill=NA, size=.75))


#Export to PDF
ggsave("output/Figure2.pdf", plot = last_plot(), 
       width = 7,
       height = 5,
       dpi = 600)


###### Figure 3 ######
#Clear environment
rm(list=ls())

#Load pegida protest spell data
load("data/daily_events.Rda")

# all Pegida protests
pegida_only <- daily_events %>% 
  filter(pro_pegida > 0) %>% 
  mutate(side = "pro-pegida") %>% 
  select(side, participants = participants_pegida_mean)

# all anti-Pegida protests
anti_pegida_only <- daily_events %>% 
  filter(anti_pegida > 0) %>% 
  mutate(side = "anti-pegida") %>% 
  select(side, participants = participants_anti_pegida_mean)

all_events <- rbind(pegida_only, anti_pegida_only) 

means <-  all_events %>%
  group_by(side) %>%
  summarise(mean_part = mean(participants, na.rm = T))

# Plot density of participant numbers

# pre define labels and ticks to obtain unlabelled smaller ticks
breaks <- seq(0, 35000, 1000)
labels <- breaks
labels[!(breaks %in% seq(0, 35000, 2000))] <- ""


ggplot(all_events, aes(x = participants)) +
  geom_histogram(aes(fill = side), col="white",
                 position = "dodge", alpha = 0.9, breaks = seq(0, 35000, 1000)) +
  geom_vline(show.legend = F, data = means, aes(xintercept = mean_part, group = side, color = side), linetype = "dashed") +
  scale_fill_manual(name = "", values = c("grey60", "grey10"), labels = c("Anti-Pegida", "Pegida"))  +
  scale_color_manual(name = "", values = c("grey60", "grey10"), labels = c("Anti-Pegida", "Pegida"))  + theme_bw() +
  theme_bw() +
  scale_x_continuous(name = "Participant numbers", breaks = breaks, label = labels) +
  scale_y_continuous(name = "Number of events", breaks = seq(0, 225, 25)) +
  theme(panel.background = element_rect(fill = "white", colour = "white"),
        panel.grid.major.x = element_blank(),
        panel.grid.minor.x = element_blank(),
        panel.grid.major.y = element_blank(),
        panel.grid.minor.y = element_blank(),
        legend.position = "bottom") +
  annotate(geom = "curve", x = 24000, y = 50, xend = 24000, yend = 2,
           curvature = .01, arrow = arrow(length = unit(.1, "mm")), lineend = "round", linetype = "dashed", color = "grey70") +
  annotate(geom = "curve", x = 25000, y = 40, xend = 25000, yend = 2, 
           curvature = .01, arrow = arrow(length = unit(.1, "mm")), lineend = "round", linetype = "dashed", color = "grey10") +
  annotate(geom = "curve", x = 30500, y = 30, xend = 30500, yend = 2, 
           curvature = .01, arrow = arrow(length = unit(.1, "mm")), lineend = "round", linetype = "dashed", color = "grey70") +
  annotate(geom = "curve", x = 33500, y = 20, xend = 33500, yend = 2, 
           curvature = .01, arrow = arrow(length = unit(.1, "mm")), lineend = "round", linetype = "dashed", color = "grey70") +
  annotate(geom = "text", x = 24000, y = 52, label = "Dresden (Jan. 2015)", hjust = "left", angle = 90, color = "grey70") +
  annotate(geom = "text", x = 25000, y = 42, label = "Dresden (Jan. 2015)", hjust = "left", angle = 90, color = "grey10") +
  annotate(geom = "text", x = 30500, y = 32, label = "Leipzig (Jan. 2015)", hjust = "left", angle = 90, color = "grey70") +
  annotate(geom = "text", x = 33500, y = 22, label = "Dresden (Jan. 2015)", hjust = "left", angle = 90, color = "grey70") +
  annotate(geom = "text", x = 2000, y = 235,  label = "mu * ' = 1,812'", hjust = "left", parse = T, color = "grey10") +
  annotate(geom = "text", x = 2700, y = 200,  label = "mu * ' = 2,433'", hjust = "left", parse = T, color = "grey70") 

#Export to PDF
ggsave("output/Figure3.pdf", plot = last_plot(), 
       width = 8,
       height = 6,
       limitsize = TRUE,
       dpi = 600)


###### Figure 4 ######

#Load pegida protest spell data
load("data/pegida_history.Rda")

# Plot survival curves for all cities 
#constant only
kmsurvival1_1 <- survfit(Surv(start, stop, pegida_onset) ~1, data = pegida_history)
#summary(kmsurvival1_1)

ggsurvplot(
  kmsurvival1_1, 
  data = pegida_history, 
  size = 1,                 # change line size
  palette = c("black"),# custom color palettes
  conf.int = TRUE,          # Add confidence interval              # Add p-value
  ggtheme = theme_bw(),
  ylim = c(0.5,1),
  xlim = c(0,100),
  legend=("none"),
  censor.shape=124,
  censor=F,
  censor.size = 4,
  risk.table = F,
  xlab = c("Time since October 2014 (in weeks)") # Change ggplot2 theme
)

#Export to PDF
ggsave("output/Figure4.pdf", plot = last_plot(), 
       width = 7,
       height = 5,
       limitsize = TRUE,
       dpi = 600)

####### Figure A1 #####

#For each city, sort events that pro-Pegida follows anti-Pegida, calculate time differences and filter for events where the last event was indeed a counter-demo.
# Then calculate the difference in participant number at Pegida demos
load("data/pegpad_waves.Rda")

#subset data to weeks where a Pegida demo took place and there was a counter-demo last week
waves_sub <- pegida_waves_final_rep %>% 
  filter(!is.na(pegida_participants_diffs) & anti_pegida_demos_lag > 0)

ggplot(data = waves_sub, 
       aes(y = I(log(participants_pegida_total_mean)),
           x = I(log(participants_anti_total_mean_lag)), 
           group=ost, shape=ost, linetype=ost)) +
  geom_point() +
  scale_shape_manual("Region", values = c(1,2)) +
  scale_linetype_manual("Region", values = c(1,3)) +
  stat_smooth(method = "lm",se=T, color = "black") + scale_y_log10(breaks = seq(3,11,2)) + scale_x_log10(breaks = seq(3,11,2)) +
  ylab("Pegida participants (log)") + xlab("Anti-Pegida articipants (t-1, log)") +
  theme(legend.title=element_blank(), strip.text = element_blank(), panel.border = element_rect(colour = "black", fill=NA, size=.5), 
        panel.grid.major = element_line(colour="lightgrey", size=0.5), panel.background = element_rect(colour = "white", fill = "white")) 

#Export to PDF
ggsave("output/FigureA1.pdf", plot = last_plot(), 
       width = 7,
       height = 5,
       dpi = 600)






